//---------------------------------------------------------------------------

#include <vcl.h>
#pragma hdrstop

#include "UnitMain.h"
#include "UnitCompile.h"
#include "UnitImportWav.h"

//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TFormMain *FormMain;


typedef unsigned char u8;
typedef unsigned short u16;

#include "zymosis.h"

#include "waveout.h"
#include "player_code.h"
#include "player_text.h"
#include "test.h"




void WrWordLH(unsigned char *data,unsigned short val)
{
	*data++=(val)&0xff;
	*data=(val>>8)&0xff;
}



void WrDWordLH(unsigned char *data,unsigned int val)
{
	*data++=(val)&0xff;
	*data++=(val>>8)&0xff;
	*data++=(val>>16)&0xff;
	*data=(val>>24)&0xff;
}



void __fastcall TFormMain::CompileAssembly(AnsiString filename,int adr,bool player)
{
	blockStruct *block;
	FILE *file;
	int i,j,k,len,off,sample;

	file=fopen(filename.c_str(),"wt");

	if(!file) return;

	if(player)
	{
		fprintf(file,"\torg %i\n\n",adr);

		for(i=0;i<player_size;++i)
		{
			if(player_data[i]!=0x0d) fputc(player_data[i],file);
		}

		fprintf(file,"\n\n");
	}

	fprintf(file,"SoundEffectsData:\n");

	for(i=0;i<effectsCount;i++)
	{
		fprintf(file,"\tdefw SoundEffect%iData\n",i);
	}

	fprintf(file,"\n");

	for(i=0;i<effectsCount;++i)
	{
		fprintf(file,"SoundEffect%iData:\n",i);

		for(j=0;j<effectsList[i].blocksCount;++j)
		{
			block=&effectsList[i].block[j];

			switch(block->type)
			{
			case 0://tone
				fprintf(file,"\tdefb 1 ;tone\n");
				fprintf(file,"\tdefw %i,%i,%i,%i,%i\n",block->frames&0xffff,block->frameLen&0xffff,block->tonePitch&0xffff,block->tonePitchSlide&0xffff,(block->duty&0xff)|((block->dutySlide&0xff)<<8));
				break;

			case 1://noise
				fprintf(file,"\tdefb 2 ;noise\n");
				fprintf(file,"\tdefw %i,%i,%i\n",block->frames&0xffff,block->frameLen&0xffff,(block->noisePitch&0xff)|((block->noisePitchSlide&0xff)<<8));
				break;

			case 2://pause
				fprintf(file,"\tdefb 1 ;pause\n");
				fprintf(file,"\tdefw %i,%i,%i,%i,%i\n",block->frames&0xffff,block->frameLen&0xffff,0,0,0);
				break;

			case 3://sample
				sample=block->sampleNumber;
				len=sampleList[sample].length;
				off=block->sampleOffset;
				if(off>=len) off=len-1;
				len-=off;

				fprintf(file,"\tdefb 3 ;sample\n");
				fprintf(file,"\tdefw %i\n",len);
				fprintf(file,"\tdefw Sample%iData+%i\n",sample,off);
				fprintf(file,"\tdefb %i\n",block->samplePitch);
				break;
			}
		}

		fprintf(file,"\tdefb 0\n");
	}

	for(i=0;i<SAMPLES_MAX;++i)
	{
		len=sampleList[i].length;

		if(!len) continue;

		fprintf(file,"\nSample%iData:\n",i);
		fprintf(file,"\tdefb ");

		for(j=0;j<len;++j)
		{
			fprintf(file,"%i",sampleList[i].data[j]);

			if(j==len-1)
			{
				fprintf(file,"\n");
			}
			else
			{
				if((j&15)==15) fprintf(file,"\n\tdefb "); else fprintf(file,",");
			}
		}
	}

	fclose(file);
}



void __fastcall TFormMain::PlayerReloc(int plradr,int sfxadr)
{
	const int off[]={ 0x1e,0x21,0x24,0x73,0x8d,0x93,0 };
	const int base=0x8000;
	int i,adr,pp;

	memcpy(&mem[plradr],player_code_data,PLAYER_SIZE);

	mem[plradr+3]=sfxadr&0xff;
	mem[plradr+4]=sfxadr>>8;

	i=0;

	while(1)
	{
		if(!off[i]) break;

		pp=plradr+off[i++];
		adr=mem[pp]+mem[pp+1]*256;
		adr=adr-base+plradr;
		mem[pp+0]=adr&0xff;
		mem[pp+1]=adr>>8;
	}
}



int __fastcall TFormMain::CompileMemBlock(int pd,int fx,int id)
{
	blockStruct *block;
	int adr,off,len;

	block=&effectsList[fx].block[id];

	switch(block->type)
	{
	case 0://tone
		{
			mem[pd++]=1;
			mem[pd++]= block->frames&0xff;
			mem[pd++]=(block->frames>>8)&0xff;
			mem[pd++]= block->frameLen&0xff;
			mem[pd++]=(block->frameLen>>8)&0xff;
			mem[pd++]= block->tonePitch&0xff;
			mem[pd++]=(block->tonePitch>>8)&0xff;
			mem[pd++]= block->tonePitchSlide&0xff;
			mem[pd++]=(block->tonePitchSlide>>8)&0xff;
			mem[pd++]= block->duty&0xff;
			mem[pd++]= block->dutySlide&0xff;
		}
		break;

	case 1://noise
		{
			mem[pd++]=2;
			mem[pd++]= block->frames&0xff;
			mem[pd++]=(block->frames>>8)&0xff;
			mem[pd++]= block->frameLen&0xff;
			mem[pd++]=(block->frameLen>>8)&0xff;
			mem[pd++]= block->noisePitch&0xff;
			mem[pd++]= block->noisePitchSlide&0xff;
		}
		break;

	case 2://pause
		{
			mem[pd++]=1;
			mem[pd++]= block->frames&0xff;
			mem[pd++]=(block->frames>>8)&0xff;
			mem[pd++]= block->frameLen&0xff;
			mem[pd++]=(block->frameLen>>8)&0xff;
			mem[pd++]=0;
			mem[pd++]=0;
			mem[pd++]=0;
			mem[pd++]=0;
			mem[pd++]=0;
			mem[pd++]=0;
		}
		break;

	case 3://sample
		{
			len=sampleList[block->sampleNumber].length;

			if(!len)
			{
				len=1;
				adr=0;
			}
			else
			{
				adr=sampleList[block->sampleNumber].adr;
				off=block->sampleOffset;
				if(off>=len) off=len-1;
				len-=off;
				adr+=off;
			}

			mem[pd++]=3;
			mem[pd++]= len&0xff;
			mem[pd++]=(len>>8)&0xff;
			mem[pd++]= adr&0xff;
			mem[pd++]=(adr>>8)&0xff;
			mem[pd++]= block->samplePitch;
		}
		break;
	}

	return pd;
}



int __fastcall TFormMain::CompileSamplesMem(int adr)
{
	bool list[SAMPLES_MAX];
	int i,len;

	CheckSamplesUse(list);

	len=0;

	for(i=0;i<SAMPLES_MAX;++i)
	{
		if(!list[i]) continue;

		sampleList[i].adr=adr;

		memcpy(&mem[adr],sampleList[i].data,sampleList[i].length);

		adr+=sampleList[i].length;
		len+=sampleList[i].length;
	}

	return len;
}



int __fastcall TFormMain::CompileMem(int adr,bool player,int block)
{
	int fxadr[EFFECTS_MAX];
	int i,j,pp,pd,slen;

	if(player)
	{
		pd=adr+PLAYER_SIZE;
		PlayerReloc(adr,pd);
	}
	else
	{
		pd=adr;
	}

	if(block<0)
	{
		pp=pd+effectsCount*2;

		for(i=0;i<effectsCount;++i)
		{
			fxadr[i]=pp;
			mem[pd++]=pp&0xff;
			mem[pd++]=pp>>8;
			pp+=CalcEffectSize(i);
		}

		slen=CompileSamplesMem(pp);

		for(i=0;i<effectsCount;++i)
		{
			for(j=0;j<effectsList[i].blocksCount;++j)
			{
				pd=CompileMemBlock(pd,i,j);
			}

			mem[pd++]=0;
		}

		pd+=slen;
	}
	else
	{
		pp=pd+2;
		mem[pd++]=pp&0xff;
		mem[pd++]=pp>>8;

		i=ListBoxEffects->ItemIndex;

		if(i>=0&&i<effectsCount)
		{
			slen=CompileSamplesMem(pp+CalcEffectSize(i));

			pd=CompileMemBlock(pd,i,block);
		}
		else
		{
			slen=0;
		}

		mem[pd++]=0;

		pd+=slen;
	}

	return pd-adr;
}



void __fastcall TFormMain::CompileBinary(AnsiString filename,int adr,bool player)
{
	FILE *file;
	int size;

	size=CompileMem(adr,player,-1);

	file=fopen(filename.c_str(),"wb");

	if(file)
	{
		fwrite(&mem[adr],size,1,file);
		fclose(file);
	}
}



void __fastcall TFormMain::SaveCodeToTap(FILE *file,int adr,int size)
{
	unsigned char tap[24];
	int i,csum;

	tap[0]=19;//size of header
	tap[1]=0;

	tap[2]=0;//flag
	tap[3]=3;//code type
	memcpy(&tap[4],"sound code",10);//name
	tap[14]=size&0xff;//size of data
	tap[15]=size>>8;
	tap[16]=adr&0xff;//parameter 1
	tap[17]=adr>>8;
	tap[18]=0;//parameter 2
	tap[19]=0;

	csum=0;
	for(i=2;i<20;i++) csum^=tap[i];

	tap[20]=csum;//checksum of the header

	tap[21]=(size+2)&255;//size of data
	tap[22]=(size+2)>>8;
	tap[23]=0xff;//flag

	fwrite(tap,sizeof(tap),1,file);
	fwrite(&mem[adr],size,1,file);

	csum=0xff;
	for(i=0;i<size;i++) csum^=mem[adr+i];

	fputc(csum,file);
}



void __fastcall TFormMain::CompileTap(AnsiString filename,int adr,bool player,bool test)
{
	FILE *file;
	int size;

	file=fopen(filename.c_str(),"wb");

	if(file)
	{
		if(!player) test=false;

		if(test)
		{
			fwrite(test_data,test_size,1,file);
			adr=32768;
		}

		size=CompileMem(adr,player,-1);

		SaveCodeToTap(file,adr,size);

		fclose(file);
	}
}



static void z80MemWrite (Z80Info *z80,uint16_t addr,uint8_t value,Z80MemIOType mio)
{
	FormMain->mem[addr]=value;
}



static uint8_t z80MemRead (Z80Info *z80,uint16_t addr,Z80MemIOType mio)
{
	return FormMain->mem[addr];
}



static uint8_t z80PortIn (Z80Info *z80,uint16_t port,Z80PIOType pio)
{
	return 0;
}



static void z80PortOut (Z80Info *z80,uint16_t port,uint8_t value,Z80PIOType pio)
{
	if((port&0xff)==0xfe) FormMain->stateSound=(value&16)?true:false; else FormMain->stateRun=false;
}



void __fastcall TFormMain::GenerateSound(int fx,int block)
{
	const unsigned char startup[]={ 0xf3,0x31,0xff,0x7f,0xcd,0x00,0x80,0xd3,0x00,0x18,0xfe };//di, set stack, call player, stop
	int i,pp,rate,smpall,ssize,size,duration,cycles,out;
	Z80Info z80;

	stateSound=false;
	stateRun=true;

	Z80_ResetCallbacks(&z80);

	z80.memReadFn =z80MemRead;
	z80.memWriteFn=z80MemWrite;
	z80.portInFn  =z80PortIn;
	z80.portOutFn =z80PortOut;
	z80.evenM1    =false;

	Z80_Reset(&z80);

	if(block>=0) fx=0;

	memcpy(mem,startup,sizeof(startup));
	CompileMem(32768,true,block);

	mem[32769]=fx;

	pp=0;

	cycles=20;
	smpall=0;

	while(stateRun&&smpall<44100*30)
	{
		out=0;

		for(i=0;i<4;i++)
		{
			cycles=20+(20-Z80_ExecuteTS(&z80,cycles));
			out+=stateSound?16384:0;
		}

		waveTest[pp++]=(out>>2);

		if(pp>=((waveTestBufSize-16)>>1))
		{
			waveTestBufSize+=1024*1024;
			waveTest=(short*)realloc(waveTest,waveTestBufSize);
		}

		smpall++;
	}

	waveTestSize=pp*2;
}



void __fastcall TFormMain::ClearBlock(int i,int j)
{
	blockStruct *block;

	block=&effectsList[i].block[j];

	memset(block,0,sizeof(blockStruct));

	block->frames=10;
	block->frameLen=1000;
	block->tonePitch=200;
	block->tonePitchSlide=0;
	block->noisePitch=100;
	block->noisePitchSlide=0;
	block->duty=128;
	block->dutySlide=0;
	block->sampleNumber=0;
	block->samplePitch=10;
	block->sampleOffset=0;
}



void __fastcall TFormMain::ClearEffect(int i)
{
	int j;

	if(i<0||i>=EFFECTS_MAX) return;

	for(j=0;j<BLOCKS_MAX;j++) ClearBlock(i,j);
	effectsList[i].blocksCount=1;

	sprintf(effectsList[i].name,"New effect %i",effectsAddCount);
	effectsAddCount++;
}



void __fastcall TFormMain::ClearAllEffects(void)
{
	int i;

	for(i=0;i<EFFECTS_MAX;++i)
	{
		effectsAddCount=0;
		ClearEffect(i);
	}

	effectsCount=1;
}



void __fastcall TFormMain::ClearSample(int i)
{
	if(i<0||i>=SAMPLES_MAX) return;

	memset(&sampleList[i],0,sizeof(sampleStruct));
	strcpy(sampleList[i].name,"empty");
	sampleList[i].pitch=1;
}



void __fastcall TFormMain::ClearAllSamples(void)
{
	int i;

	for(i=0;i<SAMPLES_MAX;++i) ClearSample(i);
}



void __fastcall TFormMain::UpdateEffectsList(int cur)
{
	int i;

	if(cur<0) cur=ListBoxEffects->ItemIndex;

	ListBoxEffects->Clear();

	for(i=0;i<effectsCount;i++)
	{
		ListBoxEffects->Items->Add(IntToStr(i)+":"+effectsList[i].name);
	}

	ListBoxEffects->ItemIndex=cur;
}



AnsiString __fastcall TFormMain::GetBlockInfo(int fx,int i)
{
	AnsiString str;
	blockStruct *block;

	block=&effectsList[fx].block[i];

	switch(block->type)
	{
	case 0:
		str="Tone ";
		str+=IntToStr(block->frames)+",";
		str+=IntToStr(block->frameLen)+",";
		str+=IntToStr(block->tonePitch)+",";
		str+=IntToStr(block->tonePitchSlide)+",";
		str+=IntToStr(block->duty)+",";
		str+=IntToStr(block->dutySlide);
		break;

	case 1:
		str="Noise ";
		str+=IntToStr(block->frames)+",";
		str+=IntToStr(block->frameLen)+",";
		str+=IntToStr(block->noisePitch)+",";
		str+=IntToStr(block->noisePitchSlide);
		break;

	case 2:
		str="Pause ";
		str+=IntToStr(block->frames)+",";
		str+=IntToStr(block->frameLen);
		break;

	case 3:
		str="Sample ";
		str+=IntToStr(block->sampleNumber)+",";
		str+=IntToStr(block->sampleOffset)+",";
		str+=IntToStr(block->samplePitch);
		break;

	default:
		str="";
	}

	return str;
}



void __fastcall TFormMain::UpdateBlocksList(int cur)
{
	int i,fx;

	fx=ListBoxEffects->ItemIndex;

	if(fx<0||fx>=effectsCount) return;

	if(cur<0) cur=ListBoxBlocks->ItemIndex;

	ListBoxBlocks->Clear();

	for(i=0;i<effectsList[fx].blocksCount;i++)
	{
		ListBoxBlocks->Items->Add(GetBlockInfo(fx,i));
	}

	ListBoxBlocks->ItemIndex=cur;
}



void __fastcall TFormMain::UpdateSamplesList(int cur)
{
	int i;
	bool list[SAMPLES_MAX];
	AnsiString str;

	CheckSamplesUse(list);

	for(i=0;i<SAMPLES_MAX;i++)
	{
		str=IntToStr(i)+":"+sampleList[i].name;
		if(list[i]) str+=" *";
		ListBoxSamples->Items->Strings[i]=str;
	}

	if(cur>=0) ListBoxSamples->ItemIndex=cur;

	GroupBoxSamples->Visible=(PageControlBlockType->ActivePage==TabSheetSample);
}



void __fastcall TFormMain::UpdateEffectName(void)
{
	int cur;

	cur=ListBoxEffects->ItemIndex;

	if(cur>=0&&cur<effectsCount)
	{
		EditEffectName->Text=effectsList[cur].name;
		EditEffectName->Enabled=true;
	}
	else
	{
		EditEffectName->Text="";
		EditEffectName->Enabled=false;
	}
}



void __fastcall TFormMain::UpdateBlockListInfo(void)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		if(block>=0&&block<effectsList[fx].blocksCount)
		{
			ListBoxBlocks->Items->Strings[block]=GetBlockInfo(fx,block);
		}
	}
}



void __fastcall TFormMain::UpdateBlockParams(void)
{
	blockStruct *block;
	int i,fx;

	fx=ListBoxEffects->ItemIndex;

	if(fx<0||fx>=effectsCount)
	{
		PageControlBlockType->Visible=false;
		return;
	}

	i=ListBoxBlocks->ItemIndex;

	if(i<0||i>=effectsList[fx].blocksCount)
	{
		PageControlBlockType->Visible=false;
		return;
	}

	PageControlBlockType->Visible=true;

	block=&effectsList[fx].block[i];

	switch(block->type)
	{
	case 0: PageControlBlockType->ActivePage=TabSheetTone;  	break;
	case 1: PageControlBlockType->ActivePage=TabSheetNoise; 	break;
	case 2: PageControlBlockType->ActivePage=TabSheetPause;		break;
	case 3: PageControlBlockType->ActivePage=TabSheetSample;	break;
	}

	EditToneFrames     ->Text=IntToStr(block->frames);
	EditToneFrameLen   ->Text=IntToStr(block->frameLen);
	EditTonePitch      ->Text=IntToStr(block->tonePitch);
	EditTonePitchSlide ->Text=IntToStr(block->tonePitchSlide);
	EditToneDuty       ->Text=IntToStr(block->duty);
	EditToneDutySlide  ->Text=IntToStr(block->dutySlide);

	EditNoiseFrames    ->Text=IntToStr(block->frames);
	EditNoiseFrameLen  ->Text=IntToStr(block->frameLen);
	EditNoisePitch     ->Text=IntToStr(block->noisePitch);
	EditNoisePitchSlide->Text=IntToStr(block->noisePitchSlide);

	EditPauseFrames    ->Text=IntToStr(block->frames);
	EditPauseFrameLen  ->Text=IntToStr(block->frameLen);

	EditSamplePitch    ->Text=IntToStr(block->samplePitch);
	EditSampleOffset   ->Text=IntToStr(block->sampleOffset);

	ListBoxSamples->ItemIndex=block->sampleNumber;
}



int __fastcall TFormMain::CalcEffectSize(int i)
{
	int j,size;

	size=1;

	for(j=0;j<effectsList[i].blocksCount;j++)
	{
		switch(effectsList[i].block[j].type)
		{
		case 0:
		case 2:
			size+=11;
			break;

		case 1:
			size+=7;
			break;

		case 3:
			size+=6;
			break;
		}
	}

	return size;
}



void __fastcall TFormMain::CheckSamplesUse(bool *list)
{
	int i,j;

	for(i=0;i<SAMPLES_MAX;++i) list[i]=false;

	for(i=0;i<effectsCount;++i)
	{
		for(j=0;j<effectsList[i].blocksCount;++j)
		{
			if(effectsList[i].block[j].type==3) list[effectsList[i].block[j].sampleNumber]=true;
		}
	}
}



int __fastcall TFormMain::CalcDataSize(void)
{
	int i,j,size;
	bool list[SAMPLES_MAX];

	size=0;

	for(i=0;i<effectsCount;++i) size=size+CalcEffectSize(i)+3;

	CheckSamplesUse(list);

	for(i=0;i<SAMPLES_MAX;++i) if(list[i]) size+=sampleList[i].length;

	return size;
}



void __fastcall TFormMain::UpdateInfo(void)
{
	int fx,size;

	fx=ListBoxEffects->ItemIndex;

	LabelEffectsCount->Caption="Effects total: "+IntToStr(effectsCount);

	LabelEffectSize->Caption="Effect size: "+((fx>=0&&fx<effectsCount)?IntToStr(CalcEffectSize(fx))+" bytes":AnsiString("---"));

	LabelSampleSize->Caption="Sample size: "+IntToStr(sampleList[ListBoxSamples->ItemIndex].length)+" bytes";

	size=CalcDataSize();
	LabelDataSize->Caption="Data size: "+IntToStr(size)+" bytes ("+IntToStr(size+PLAYER_SIZE)+" bytes with player)";
}



void __fastcall TFormMain::UpdateAddDel(void)
{
	int fx;

	SpeedButtonEffectAdd  ->Enabled=effectsCount<EFFECTS_MAX?true:false;
	SpeedButtonEffectClone->Enabled=SpeedButtonEffectAdd->Enabled;
	SpeedButtonEffectDel  ->Enabled=effectsCount>1?true:false;

	fx=ListBoxEffects->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		SpeedButtonBlockAdd  ->Enabled=effectsList[fx].blocksCount<BLOCKS_MAX?true:false;
		SpeedButtonBlockClone->Enabled=SpeedButtonBlockAdd->Enabled;
		SpeedButtonBlockDel  ->Enabled=effectsList[fx].blocksCount>1?true:false;
	}

	GroupBoxSamples->Visible=(PageControlBlockType->ActivePage==TabSheetSample);
}



void __fastcall TFormMain::UpdateAll(int fx,int block)
{
	UpdateEffectsList(fx);
	UpdateEffectName();
	UpdateBlocksList(block);
	UpdateBlockParams();
	UpdateInfo();
	UpdateAddDel();
	UpdateSamplesList(-1);
}



void __fastcall TFormMain::NewProject(void)
{
	ClearAllEffects();
	ClearAllSamples();
	UpdateAll(0,0);
}



int __fastcall TFormMain::LoadProject(AnsiString name,void *listPtr)
{
	char str[1024];
	FILE *file;
	int i,j,n,len,count,ver;
	effectStruct *list;

	list=(effectStruct*)listPtr;

	file=fopen(name.c_str(),"rt");

	if(file)
	{
		fscanf(file,"%s",str);

		ver=0;

		if(!strcmp(str,"SoundEffectsProjectV2")) ver=2;
		if(!strcmp(str,"SoundEffectsProject"  )) ver=1;

		if(!ver)
		{
			Application->MessageBox("Wrong or corrupted file","Error",MB_OK);
			return 0;
		}

		fscanf(file,"%i",&count);

		for(i=0;i<count;i++)
		{
			fscanf(file,"%s",str);

			for(j=0;j<sizeof(str);j++) if(str[j]=='~') str[j]=' ';

			strcpy(list[i].name,str);

			fscanf(file,"%i",&list[i].blocksCount);

			for(j=0;j<list[i].blocksCount;j++)
			{
				fscanf(file,"%i",&list[i].block[j].type);
				fscanf(file,"%i",&list[i].block[j].frames);
				fscanf(file,"%i",&list[i].block[j].frameLen);
				fscanf(file,"%i",&list[i].block[j].tonePitch);
				fscanf(file,"%i",&list[i].block[j].tonePitchSlide);
				fscanf(file,"%i",&list[i].block[j].noisePitch);
				fscanf(file,"%i",&list[i].block[j].noisePitchSlide);
				fscanf(file,"%i",&list[i].block[j].duty);
				fscanf(file,"%i",&list[i].block[j].dutySlide);

				if(ver>1)
				{
					fscanf(file,"%i",&list[i].block[j].samplePitch);
					fscanf(file,"%i",&list[i].block[j].sampleNumber);
					fscanf(file,"%i",&list[i].block[j].sampleOffset);
				}
			}
		}

		if(ver>1)
		{
			for(i=0;i<SAMPLES_MAX;++i)
			{
				fscanf(file,"%i",&len);

				sampleList[i].length=len;

				if(len)
				{
					fscanf(file,"%i",&sampleList[i].pitch);

					fscanf(file,"%s",str);

					for(j=0;j<sizeof(str);j++) if(str[j]=='~') str[j]=' ';

					strcpy(sampleList[i].name,str);

					for(j=0;j<len;++j)
					{
						fscanf(file,"%i",&sampleList[i].data[j]);
					}
				}
			}
		}

		fclose(file);

		return count;
	}

	return 0;
}



bool __fastcall TFormMain::SaveProject(AnsiString name)
{
	FILE *file;
	int i,j,len;
	char buf[256];

	file=fopen(name.c_str(),"wt");

	if(!file) return false;

	fprintf(file,"SoundEffectsProjectV2 %i ",effectsCount);

	for(i=0;i<effectsCount;i++)
	{
		strcpy(buf,effectsList[i].name);

		for(j=0;j<sizeof(buf);j++) if(buf[j]==' ') buf[j]='~';

		fprintf(file,"%s %i ",buf,effectsList[i].blocksCount);

		for(j=0;j<effectsList[i].blocksCount;j++)
		{
			fprintf(file,"%i ",effectsList[i].block[j].type);
			fprintf(file,"%i ",effectsList[i].block[j].frames);
			fprintf(file,"%i ",effectsList[i].block[j].frameLen);
			fprintf(file,"%i ",effectsList[i].block[j].tonePitch);
			fprintf(file,"%i ",effectsList[i].block[j].tonePitchSlide);
			fprintf(file,"%i ",effectsList[i].block[j].noisePitch);
			fprintf(file,"%i ",effectsList[i].block[j].noisePitchSlide);
			fprintf(file,"%i ",effectsList[i].block[j].duty);
			fprintf(file,"%i ",effectsList[i].block[j].dutySlide);
			fprintf(file,"%i ",effectsList[i].block[j].samplePitch);
			fprintf(file,"%i ",effectsList[i].block[j].sampleNumber);
			fprintf(file,"%i ",effectsList[i].block[j].sampleOffset);
		}
	}

	for(i=0;i<SAMPLES_MAX;++i)
	{
		len=sampleList[i].length;

		fprintf(file,"%i ",len);

		if(!len) continue;

		fprintf(file,"%i ",sampleList[i].pitch);

		strcpy(buf,sampleList[i].name);

		for(j=0;j<sizeof(buf);j++) if(buf[j]==' ') buf[j]='~';

		fprintf(file,"%s ",buf);

		for(j=0;j<len;++j) fprintf(file,"%i ",sampleList[i].data[j]);
	}

	fclose(file);

	return true;
}

//---------------------------------------------------------------------------
__fastcall TFormMain::TFormMain(TComponent* Owner)
: TForm(Owner)
{
}
//---------------------------------------------------------------------------
void __fastcall TFormMain::MCloseClick(TObject *Sender)
{
	if(Application->MessageBox("Exit?","Confirm",MB_YESNO)==IDYES) Close();
}
//---------------------------------------------------------------------------
void __fastcall TFormMain::ListBoxEffectsDblClick(TObject *Sender)
{
	MPlayEffectClick(Sender);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MPlayEffectClick(TObject *Sender)
{
	int fx;

	fx=ListBoxEffects->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		SoundStop();
		GenerateSound(fx,-1);
		SoundPlay(Handle,waveTest,waveTestSize);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MPlayBlockClick(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		if(block>=0&&block<effectsList[fx].blocksCount)
		{
			SoundStop();
			GenerateSound(fx,block);
			SoundPlay(Handle,waveTest,waveTestSize);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MStopClick(TObject *Sender)
{
	SoundStop();
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::FormCreate(TObject *Sender)
{
	AnsiString name;
	FILE *file;
	int i;

	for(i=0;i<16384;i++) mem[i]=rand();

	srand(timeGetTime());

	name=ParamStr(0).SubString(1,ParamStr(0).LastDelimiter("\\/"))+"48.rom";

	if(FileExists(name))
	{
		file=fopen(name.c_str(),"rb");

		if(file)
		{
			fread(mem,16384,1,file);
			fclose(file);
		}
	}

	waveTestBufSize=1024*1024;
	waveTest=(short*)malloc(waveTestBufSize);

	NewProject();

	SoundInit();

	GroupBoxEffects->DoubleBuffered=true;
	GroupBoxBlocks ->DoubleBuffered=true;
	GroupBoxSamples->DoubleBuffered=true;
	ListBoxSamples ->DoubleBuffered=true;
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::FormDestroy(TObject *Sender)
{
	SoundStop();

	if(waveTest) free(waveTest);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MNewClick(TObject *Sender)
{
	if(Application->MessageBox("New project?","Confirm",MB_YESNO)==IDYES)
	{
		NewProject();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonEffectAddClick(TObject *Sender)
{
	if(effectsCount<EFFECTS_MAX)
	{
		ClearEffect(effectsCount);
		effectsCount++;
		UpdateAll(effectsCount-1,0);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonEffectDelClick(TObject *Sender)
{
	int cur;

	cur=ListBoxEffects->ItemIndex;

	if(cur>=0&&cur<effectsCount&&effectsCount>1)
	{
		if(Application->MessageBox("Remove effect?","Confirm",MB_YESNO)==IDYES)
		{
			memcpy(&effectsList[cur],&effectsList[cur+1],sizeof(effectStruct)*(effectsCount-cur));
			effectsCount--;
			if(cur>effectsCount-1) cur=effectsCount-1;
			UpdateAll(cur,0);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonEffectUpClick(TObject *Sender)
{
	int cur;
	effectStruct temp;

	cur=ListBoxEffects->ItemIndex;

	if(cur>0)
	{
		memcpy(&temp,&effectsList[cur-1],sizeof(effectStruct));
		memcpy(&effectsList[cur-1],&effectsList[cur],sizeof(effectStruct));
		memcpy(&effectsList[cur],&temp,sizeof(effectStruct));
		UpdateEffectsList(cur-1);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonEffectDownClick(TObject *Sender)
{
	int cur;
	effectStruct temp;

	cur=ListBoxEffects->ItemIndex;

	if(cur<effectsCount-1)
	{
		memcpy(&temp,&effectsList[cur+1],sizeof(effectStruct));
		memcpy(&effectsList[cur+1],&effectsList[cur],sizeof(effectStruct));
		memcpy(&effectsList[cur],&temp,sizeof(effectStruct));
		UpdateEffectsList(cur+1);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MOpenProjectClick(TObject *Sender)
{
	int count;

	OpenDialogProject->Title="Open project";

	if(OpenDialogProject->Execute())
	{
		ClearAllSamples();

		count=LoadProject(OpenDialogProject->FileName,effectsList);

		if(count)
		{
			effectsCount=count;

			SaveDialogProject->FileName=OpenDialogProject->FileName;
			FormCompile->SaveDialogData->FileName=OpenDialogProject->FileName;

			UpdateAll(0,0);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MSaveProjectClick(TObject *Sender)
{
	if(SaveDialogProject->Execute())
	{
		SaveProject(SaveDialogProject->FileName);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxEffectsClick(TObject *Sender)
{
	UpdateEffectName();
	UpdateBlocksList(0);
	UpdateBlockParams();
	UpdateAddDel();
	UpdateInfo();
	UpdateSamplesList(-1);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxEffectsKeyUp(TObject *Sender, WORD &Key,
TShiftState Shift)
{
	ListBoxEffectsClick(Sender);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditEffectNameChange(TObject *Sender)
{
	int cur;

	cur=ListBoxEffects->ItemIndex;

	if(cur>=0&&cur<effectsCount)
	{
		strcpy(effectsList[cur].name,EditEffectName->Text.c_str());
		ListBoxEffects->Items->Strings[cur]=IntToStr(cur)+":"+EditEffectName->Text;
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonEffectCloneClick(TObject *Sender)
{
	int i,cur;

	cur=ListBoxEffects->ItemIndex;

	if(cur>=0&&cur<effectsCount&&effectsCount<EFFECTS_MAX)
	{
		for(i=effectsCount;i>cur;i--)
		{
			memcpy(&effectsList[i],&effectsList[i-1],sizeof(effectStruct));
		}

		effectsCount++;
		UpdateAll(cur+1,0);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MCompileClick(TObject *Sender)
{
	if(CalcDataSize()>32768)
	{
		Application->MessageBox("Data size is too large to compile (32768 byres max)","Error",MB_OK);
	}
	else
	{
		FormCompile->PlayerSize=PLAYER_SIZE;
		FormCompile->ShowModal();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonBlockAddClick(TObject *Sender)
{
	int i;

	i=ListBoxEffects->ItemIndex;

	if(i>=0&&i<effectsCount)
	{
		if(effectsList[i].blocksCount<128)
		{
			ClearBlock(i,effectsList[i].blocksCount);
			effectsList[i].blocksCount++;
			UpdateAll(-1,effectsList[i].blocksCount-1);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxBlocksClick(TObject *Sender)
{
	UpdateBlockParams();
	UpdateAddDel();
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxBlocksKeyUp(TObject *Sender, WORD &Key,
TShiftState Shift)
{
	ListBoxBlocksClick(Sender);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::PageControlBlockTypeChange(TObject *Sender)
{
	int fx,block,type;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	type=-1;
	if(PageControlBlockType->ActivePage==TabSheetTone)   type=0;
	if(PageControlBlockType->ActivePage==TabSheetNoise)  type=1;
	if(PageControlBlockType->ActivePage==TabSheetPause)  type=2;
	if(PageControlBlockType->ActivePage==TabSheetSample) type=3;

	if(type>=0)
	{
		if(fx>=0&&fx<effectsCount)
		{
			if(block>=0&&block<effectsList[fx].blocksCount)
			{
				effectsList[fx].block[block].type=type;

				UpdateBlocksList(block);
				UpdateSamplesList(effectsList[fx].block[block].sampleNumber);
			}
		}
	}

	UpdateInfo();
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonBlockDelClick(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		if(block>=0&&block<effectsList[fx].blocksCount)
		{
			if(Application->MessageBox("Remove block?","Confirm",MB_YESNO)==IDYES)
			{
				memcpy(&effectsList[fx].block[block],&effectsList[fx].block[block+1],sizeof(blockStruct)*(effectsList[fx].blocksCount-block));
				effectsList[fx].blocksCount--;
				if(block>effectsList[fx].blocksCount-1) block=effectsList[fx].blocksCount-1;
				UpdateAll(-1,block);
			}
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonBlockCloneClick(TObject *Sender)
{
	int i,fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount&&effectsCount<EFFECTS_MAX)
	{
		if(block>=0&&block<effectsList[fx].blocksCount)
		{
			for(i=effectsList[fx].blocksCount;i>block;i--)
			{
				memcpy(&effectsList[fx].block[i],&effectsList[fx].block[i-1],sizeof(blockStruct));
			}

			effectsList[fx].blocksCount++;
			UpdateAll(-1,block+1);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonBlockUpClick(TObject *Sender)
{
	int fx,block;
	blockStruct temp;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		if(block>0)
		{
			memcpy(&temp,&effectsList[fx].block[block-1],sizeof(blockStruct));
			memcpy(&effectsList[fx].block[block-1],&effectsList[fx].block[block],sizeof(blockStruct));
			memcpy(&effectsList[fx].block[block],&temp,sizeof(blockStruct));
			UpdateBlocksList(block-1);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonBlockDownClick(TObject *Sender)
{
	int fx,block;
	blockStruct temp;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(fx>=0&&fx<effectsCount)
	{
		if(block<effectsList[fx].blocksCount-1)
		{
			memcpy(&temp,&effectsList[fx].block[block+1],sizeof(blockStruct));
			memcpy(&effectsList[fx].block[block+1],&effectsList[fx].block[block],sizeof(blockStruct));
			memcpy(&effectsList[fx].block[block],&temp,sizeof(blockStruct));
			UpdateBlocksList(block+1);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneFramesKeyPress(TObject *Sender, char &Key)
{
	if(((TEdit*)Sender)->Tag)
	{
		if(Key!=VK_BACK&&Key!='-'&&(Key<'0'||Key>'9')) Key=0;
	}
	else
	{
		if(Key!=VK_BACK&&(Key<'0'||Key>'9')) Key=0;
	}
}
//---------------------------------------------------------------------------



void __fastcall TFormMain::EditToneFramesChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<1) val=1;
		if(val>65536) val=65536;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].frames=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneFramesExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].frames);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneFrameLenChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<1) val=1;
		if(val>65536) val=65536;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].frameLen=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneFrameLenExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].frameLen);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditTonePitchChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<1) val=1;
		if(val>65535) val=65535;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].tonePitch=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditTonePitchExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].tonePitch);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditTonePitchSlideChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-100000);

	if(val>-100000)
	{
		if(val<-32767) val=-32767;
		if(val> 32768) val=32768;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].tonePitchSlide=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditTonePitchSlideExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(StrToIntDef(((TEdit*)Sender)->Text,-100000)==-100000) ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].tonePitchSlide);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditNoisePitchChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<1) val=1;
		if(val>256) val=256;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].noisePitch=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditNoisePitchExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].noisePitch);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditNoisePitchSlideChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-100000);

	if(val>-100000)
	{
		if(val<-127) val=-127;
		if(val> 128) val=128;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].noisePitchSlide=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditNoisePitchSlideExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(StrToIntDef(((TEdit*)Sender)->Text,-100000)==-100000) ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].noisePitchSlide);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneDutyChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<0) val=0;
		if(val>255) val=255;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].duty=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneDutyExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].duty);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneDutySlideChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-1000);

	if(val>-1000)
	{
		if(val<-127) val=-127;
		if(val> 128) val=128;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].dutySlide=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditToneDutySlideExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(StrToIntDef(((TEdit*)Sender)->Text,-100000)==-100000) ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].dutySlide);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MRandomizeBlockClick(TObject *Sender)
{
	blockStruct *block;
	int fx,id,flags;

	if(Application->MessageBox("Randomize parameters?","Confirm",MB_YESNO)==IDYES)
	{
		fx=ListBoxEffects->ItemIndex;
		id=ListBoxBlocks->ItemIndex;

		if(fx>=0&&fx<effectsCount)
		{
			if(id>=0&&id<effectsList[fx].blocksCount)
			{
				block=&effectsList[fx].block[id];

				flags=rand();

				block->frames=1+(rand()%1000);
				block->frameLen=1+(rand()%10000);

				block->tonePitch=1+(rand()%(flags&1?1000:65535));
				block->noisePitch=1+(rand()&255);

				if(!(flags&128))
				{
					block->tonePitchSlide=rand()&(flags&64?65535:100);
					block->noisePitchSlide=0;
				}
				else
				{
					block->tonePitchSlide=0;
					block->noisePitchSlide=rand()&(flags&8?10:255);
				}

				if(flags&8) block->duty=1+(rand()%255); else block->duty=128;

				if(flags&16)
				{
					block->dutySlide=0;
				}
				else
				{
					block->dutySlide=-127+(rand()&(flags&32?10:255));
				}

				block->samplePitch=1+(rand()&31);

				UpdateBlockParams();
			}
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxBlocksDblClick(TObject *Sender)
{
	MPlayBlockClick(Sender);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::MMergeProjectClick(TObject *Sender)
{
	effectStruct list[EFFECTS_MAX];
	int i,count;

	OpenDialogProject->Title="Merge project";

	if(OpenDialogProject->Execute())
	{
		count=LoadProject(OpenDialogProject->FileName,list);

		if(count)
		{
			for(i=0;i<count;i++)
			{
				if(effectsCount>=EFFECTS_MAX) break;
				memcpy(&effectsList[effectsCount++],&list[i],sizeof(effectStruct));
			}

			UpdateAll(0,0);
		}
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxSamplesClick(TObject *Sender)
{
	effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].sampleNumber=ListBoxSamples->ItemIndex;
	UpdateBlockListInfo();
	UpdateInfo();
	UpdateSamplesList(-1);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonClearSampleClick(TObject *Sender)
{
	if(Application->MessageBox("Are you sure?","Confirm",MB_YESNO)==IDYES)
	{
		ClearSample(ListBoxSamples->ItemIndex);
		UpdateAll(0,0);
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::SpeedButtonImportSampleClick(TObject *Sender)
{
	FormImportWAV->Show();
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::FormResize(TObject *Sender)
{
	GroupBoxEffects->Height=ClientHeight-GroupBoxEffects->Top;
	GroupBoxBlocks ->Height=ClientHeight-GroupBoxBlocks ->Top;
	GroupBoxSamples->Height=ClientHeight-GroupBoxSamples->Top;

	ListBoxEffects->Height=GroupBoxEffects->Height-ListBoxEffects->Top-8;
	ListBoxBlocks ->Height=GroupBoxBlocks ->Height-ListBoxBlocks ->Top-8;
	ListBoxSamples->Height=GroupBoxSamples->Height-ListBoxSamples->Top-8;
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditSamplePitchChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<1) val=1;
		if(val>255) val=255;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].samplePitch=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditSamplePitchExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].samplePitch);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditSampleOffsetChange(TObject *Sender)
{
	int val;

	val=StrToIntDef(((TEdit*)Sender)->Text,-2);

	if(val>-2)
	{
		if(val<0) val=0;
		if(val>32768) val=32768;

		((TEdit*)Sender)->Text=IntToStr(val);
		effectsList[ListBoxEffects->ItemIndex].block[ListBoxBlocks->ItemIndex].sampleOffset=val;

		UpdateBlockListInfo();
	}
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::EditSampleOffsetExit(TObject *Sender)
{
	int fx,block;

	fx=ListBoxEffects->ItemIndex;
	block=ListBoxBlocks->ItemIndex;

	if(((TEdit*)Sender)->Text=="") ((TEdit*)Sender)->Text=IntToStr(effectsList[fx].block[block].sampleOffset);
}
//---------------------------------------------------------------------------

void __fastcall TFormMain::ListBoxSamplesDblClick(TObject *Sender)
{
	effectStruct temp;
	blockStruct *block;

	memcpy(&temp,&effectsList[0],sizeof(effectStruct));

	effectsList[0].blocksCount=1;

	block=&effectsList[0].block[0];

	block->type=3;
	block->sampleNumber=ListBoxSamples->ItemIndex;
	block->sampleOffset=0;
	block->samplePitch=sampleList[block->sampleNumber].pitch;

	SoundStop();
	GenerateSound(0,-1);
	SoundPlay(Handle,waveTest,waveTestSize);

	memcpy(&effectsList[0],&temp,sizeof(effectStruct));
}
//---------------------------------------------------------------------------

